const Sha0 = require('./sha0');
const TeraCrypto = require('tera-network-crypto');

function getCrypto(key) {
    const data = Buffer.allocUnsafe(680);
    data[0] = 128;
    for (let i = 1; i < 680; i++)
        data[i] = key[i % 128];
    for (let i = 0; i < 680; i += 20) {
        const sha0 = new Sha0();
        sha0.update(data);
        const hash = sha0.hash();
        for (let j = 0; j < 20; j += 4) data.writeUInt32LE(hash.readUInt32BE(j), i + j);
    }
    return new TeraCrypto(data);
}

function shiftKey(tgt, src, n) {
    const len = src.length;
    if (n > 0) {
        src.copy(tgt, 0, n);
        src.copy(tgt, len - n);
    } else {
        src.copy(tgt, 0, len + n);
        src.copy(tgt, -n);
    }
    return tgt;
}

function xorKey(tgt, key1, key2) {
    for (let i = 0; i < 128; i++)
        tgt[i] = key1[i] ^ key2[i];
}

class Session {
    constructor(protocolVersion, majorPatchVersion) {
        this.protocolVersion = protocolVersion;
        this.majorPatchVersion = majorPatchVersion;
        this.encryptor = null;
        this.decryptor = null;
        this.useOldConstants = majorPatchVersion < 45;
        this.clientKeys = [Buffer.alloc(128), Buffer.alloc(128)];
        this.serverKeys = [Buffer.alloc(128), Buffer.alloc(128)];
    }

    init() {
        const [c1, c2] = this.clientKeys;
        const [s1, s2] = this.serverKeys;
        const t1 = Buffer.allocUnsafe(128);
        const t2 = Buffer.allocUnsafe(128);
        shiftKey(t1, s1, this.useOldConstants ? -31 : -67);
        xorKey(t2, t1, c1);
        shiftKey(t1, c2, this.useOldConstants ? 17 : 29);
        xorKey(t2, t1, t2);
        this.decryptor = getCrypto(t2);
        shiftKey(t1, s2, this.useOldConstants ? -79 : -41);
        this.decryptor.apply(t1);
        this.encryptor = getCrypto(t1);
    }

    applyFromClient(data) {
        return this.decryptor.apply(data);
    }

    applyToClient(data) {
        return this.encryptor.apply(data);
    }

    applyFromServer(data) {
        return this.encryptor.apply(data);
    }

    applyToServer(data) {
        return this.decryptor.apply(data);
    }

    cloneKeys() {
        const session = new Session(this.protocolVersion, this.majorPatchVersion);
        this.clientKeys[0].copy(session.clientKeys[0]);
        this.clientKeys[1].copy(session.clientKeys[1]);
        this.serverKeys[0].copy(session.serverKeys[0]);
        this.serverKeys[1].copy(session.serverKeys[1]);
        session.init();
        return session;
    }
}

module.exports = Session;
