const mui = require('tera-toolbox-mui').DefaultInstance;
const net = require('net');
const path = require('path');
const EventEmitter = require('events');
const GPKManager = require('./GPKManager');

class TeraClientInterfaceConnection extends EventEmitter {
    constructor(socket, GPKManager, Injector) {
        super();
        this.setMaxListeners(0);

        this.socket = socket;
        this.buffer = null;

        this.dataQueryEventEmitter = new EventEmitter;
        this.nextDataQueryId = 0;

        this.GPKManager = GPKManager;
        this.injector = Injector;

        this.socket.on('data', data => {
            if (!this.socket)
                return;

            this.buffer = this.buffer ? Buffer.concat([this.buffer, data]) : data;

            let start = 0;
            let end = -1;
            while ((end = this.buffer.indexOf(0, start)) >= 0) {
                if (end > start) {
                    const packet = this.buffer.slice(start, end);
                    try {
                        const parsed = JSON.parse(packet);
                        switch (parsed.command) {
                            case 'dcresult':
                                this.dataQueryEventEmitter.emit(parsed.data.id.toString(), parsed.data);
                                break;
                            case 'ready':
                                this.emit('data', parsed.command, parsed.data || {});
                                this.emit('ready');
                                break;
                            case 'hasfocus':
                                this.emit('hasfocus', parsed.data.result);
                                break;
                            default:
                                this.emit('data', parsed.command, parsed.data || {});
                                break;
                        }
                    } catch (e) {
                        console.log(mui.get('tera-client-interface/index/communication-error'));
                        console.log(e);
                    }
                }

                start = end + 1;
            }

            this.buffer = this.buffer.slice(start);
        });

        this.socket.once('error', e => {
            this._onClose(e);
            this.socket = null;
        });

        this.socket.once('close', () => {
            this._onClose();
            this.socket = null;
        });
    }

    send(command, data = {}) {
        if (this.socket)
            this.socket.write(JSON.stringify({ command, data }) + "\x00");
    }

    _onClose(error) {
        this.emit('disconnect', error);
        this.removeAllListeners();

        if (this.GPKManager && this.info && this.info.path)
            this.GPKManager.uninstallAll(path.join(this.info.path, '..'));
        this.GPKManager = null;

        if (this.dataQueryEventEmitter) {
            this.dataQueryEventEmitter.removeAllListeners();
            this.dataQueryEventEmitter = null;
        }
    }

    destructor() {
        if (this.socket) {
            this.socket.end();
            this.socket.destroy();
            this.socket = null;
        } else {
            this._onClose();
        }
    }

    queryData(query, queryArgs = null, findAll = false, children = true, attributeFilter = null) {
        const queryId = this.nextDataQueryId++;

        const res = new Promise((resolve, reject) => {
            if (!this.dataQueryEventEmitter) {
                reject(new Error('Connection to client closed!'));
            } else {
                this.dataQueryEventEmitter.once(queryId.toString(), result => {
                    if (result.success)
                        resolve(result.data);
                    else
                        reject(result.error);
                });
            }
        });

        this.send('dcquery', {
            id: queryId,
            query: query,
            arguments: queryArgs,
            findall: findAll,
            children: children,
            attributeFilter: attributeFilter,
        });

        return res;
    }

    flashWindow(count = 5, interval = 0, allowFocused = false) {
        if (typeof (count) !== 'number')
            throw new Error('Count must be a number!');
        if (typeof (interval) !== 'number')
            throw new Error('Interval must be a number!');
        if (typeof (allowFocused) !== 'boolean')
            throw new Error('AllowFocused must be a boolean!');

        this.send('flashwindow', { count, interval, allowFocused });
    }

    hasFocus() {
        const res = new Promise((resolve, reject) => {
            this.once('hasfocus', result => {
                resolve(result);
            });
        });

        this.send('hasfocus');
        return res;
    }

    configureCameraShake(enabled, power = 1.0, speed = 1.0) {
        if (typeof (enabled) !== 'boolean')
            throw new Error('Enabled must be a boolean!');
        if (typeof (power) !== 'number')
            throw new Error('Power must be a number!');
        if (typeof (speed) !== 'number')
            throw new Error('Speed must be a number!');

        this.send('camerashake', { enabled, power, speed });
    }

    suspend() {
        this.send('suspend', {});
    }

    resume() {
        this.send('resume', {});
    }

    installGPK(fromPath, filename = null) {
        if (!filename)
            filename = path.basename(fromPath);
        this.GPKManager.install(path.join(this.info.path, '..'), filename, fromPath, this.info.arch, this.info.majorPatchVersion, this.info.minorPatchVersion);
    }

    _installGPKs() {
        this.send('installgpks', { folder: this.GPKManager.GPKFolderName });
    }

    injectDLL(path) {
        return this.injector.inject(this.info.pid, path);
    }
}

class TeraClientInterfaceServer {
    constructor(isAdmin, host, port, onAccept, onReady, onError) {
        this.host = host;
        this.port = port;
        this.onAccept = onAccept;

        this.GPKManager = new GPKManager('_Toolbox');

        this.processListeners = new Set;
        this.connections = new Set;

        this.server = net.createServer(socket => this.accept(socket));
        this.server.on('listening', () => onReady());
        this.server.on('error', e => onError(e));

        if (process.platform === 'win32') {
            const scanner = require('./scanner');

            ['TERA.exe', 'VTEQ.exe'].forEach(executable => this.processListeners.add(
                new scanner(executable, path.join(__dirname, 'tera-client-interface-32.dll'), path.join(__dirname, 'tera-client-interface-64.dll'), 25)
            ));
        }
    }

    destructor() {
        this.processListeners.forEach(processListener => processListener.stop());
        this.processListeners.clear();

        this.connections.forEach(connection => connection.destructor());
        this.connections.clear();

        this.server.close();
        this.server = null;

        if (this.GPKManager) {
            this.GPKManager.destructor();
            this.GPKManager = null;
        }

        this.onAccept = null;
    }

    run() {
        this.server.listen(this.port, this.host);
        this.processListeners.forEach(processListener => processListener.start());
    }

    accept(socket) {
        socket.setNoDelay(true);

        const connection = new TeraClientInterfaceConnection(socket, this.GPKManager, [...this.processListeners][0]);
        this.connections.add(connection);

        socket.once('error', () => this.onDisconnect(connection));
        socket.once('close', () => this.onDisconnect(connection));

        this.onAccept(connection);

        this.processListeners.forEach(processListener => processListener.setInterval(1000));
    }

    onDisconnect(connection) {
        this.connections.delete(connection);
        this.processListeners.forEach(processListener => processListener.setInterval(this.connections.size > 0 ? 500 : 25));
    }
}

module.exports = TeraClientInterfaceServer;
