const mui = require('tera-toolbox-mui').DefaultInstance;
const path = require('path');
const fs = require('fs');
const crypto = require('crypto');

function forcedirSync(dir) {
    const sep = path.sep;
    const initDir = path.isAbsolute(dir) ? sep : '';
    dir.split(sep).reduce((parentDir, childDir) => {
        const curDir = path.resolve(parentDir, childDir);
        try {
            fs.mkdirSync(curDir);
        } catch (_) {
            // Ignore
        }

        return curDir;
    }, initDir);
}

function rmdirSyncForce(dir_path) {
    if (!fs.existsSync(dir_path))
        return;

    fs.readdirSync(dir_path).forEach(entry => {
        const entry_path = path.join(dir_path, entry);
        if (fs.lstatSync(entry_path).isDirectory()) {
            rmdirSyncForce(entry_path);
        } else {
            try {
                fs.unlinkSync(entry_path);
            } catch (_) {
                // Ignore
            }
        }
    });

    try {
        fs.rmdirSync(dir_path);
    } catch (_) {
        // Ignore
    }
}

function hash(data) {
    return crypto.createHash("sha256").update(data).digest().toString("hex").toUpperCase();
}

function isCompatible(data, arch, majorPatchVersion, minorPatchVersion) {
    const magic = data.readUInt32LE(0);
    if (magic !== 0x9E2A83C1)
        return false;

    // TODO: This is a hacky check, is there a better way to identify compatibility?
    const fileVersion = data.readUInt16LE(4);
    switch (arch) {
        case 'ia32': return fileVersion <= 610;
        case 'x64': return fileVersion >= 897;
        default: return true;
    }
}

class GPKManager {
    constructor(GPKFolderName) {
        this.GPKFolderName = GPKFolderName;
        this.installedFiles = {};
        this.hasShownSymlinkWarning = false;
    }

    destructor() {

    }

    getGPKFolderPath(clientFolder) {
        return path.join(clientFolder, 'S1Game', 'CookedPC', this.GPKFolderName);
    }

    getFullPath(clientFolder, filename) {
        return path.join(this.getGPKFolderPath(clientFolder), filename);
    }

    initialize(clientFolder) {
        if (this.hasInstalled(clientFolder))
            return;

        rmdirSyncForce(this.getGPKFolderPath(clientFolder));
    }

    hasInstalled(clientFolder) {
        const folder = this.getGPKFolderPath(clientFolder);
        return Object.keys(this.installedFiles).some(filename => filename.startsWith(folder));
    }

    getInstalled(clientFolder) {
        const folder = this.getGPKFolderPath(clientFolder);
        return Object.keys(this.installedFiles).filter(filename => filename.startsWith(folder));
    }

    install(clientFolder, filename, fromPath, clientArchitecture, clientMajorPatchVersion, clientMinorPatchVersion) {
        const fullPath = this.getFullPath(clientFolder, filename);

        const fromContents = fs.readFileSync(fromPath);
        if (!isCompatible(fromContents, clientArchitecture, clientMajorPatchVersion, clientMinorPatchVersion))
            throw new Error(`Trying to install incompatible GPK file: ${fullPath}`);

        const fromHash = hash(fromContents);

        if (this.installedFiles[fullPath]) {
            if (this.installedFiles[fullPath].hash !== fromHash)
                throw new Error(`Trying to install two different GPK files with the same name: ${fullPath}`);

            ++this.installedFiles[fullPath].count;
        } else {
            forcedirSync(path.dirname(fullPath));

            try {
                fs.symlinkSync(fromPath, fullPath, 'file');
            } catch (e) {
                switch (e.code) {
                    case 'EISDIR': {
                        if (!this.hasShownSymlinkWarning) {
                            this.hasShownSymlinkWarning = true;
                            console.log(mui.get('tera-client-interface/gpkmanager/symlink-warning-1'));
                            console.log(mui.get('tera-client-interface/gpkmanager/symlink-warning-2'));
                            console.log(mui.get('tera-client-interface/gpkmanager/symlink-warning-3'));
                            console.log(mui.get('tera-client-interface/gpkmanager/symlink-warning-4'));
                        }

                        fs.copyFileSync(fromPath, fullPath);
                        break;
                    }
                    default: {
                        throw e;
                    }
                }
            }

            this.installedFiles[fullPath] = {
                hash: fromHash,
                count: 1,
            };
        }
    }

    uninstall(clientFolder, filename) {
        this._uninstall(this.getFullPath(clientFolder, filename));
    }

    _uninstall(fullPath) {
        if (!this.installedFiles[fullPath])
            return;

        if (this.installedFiles[fullPath].count > 1) {
            --this.installedFiles[fullPath].count;
        } else {
            try {
                fs.unlinkSync(fullPath);
            } catch (e) {
                console.log(mui.get('tera-client-interface/gpkmanager/uninstall-error-1'));
                console.log(mui.get('tera-client-interface/gpkmanager/uninstall-error-2', {fullPath}));
                console.log(mui.get('tera-client-interface/gpkmanager/uninstall-error-3'));
            }
            delete this.installedFiles[fullPath];
        }
    }

    uninstallAll(clientFolder) {
        this.getInstalled(clientFolder).forEach(fullPath => this._uninstall(fullPath));
        if (!this.hasInstalled(clientFolder))
            rmdirSyncForce(this.getGPKFolderPath(clientFolder));
    }
}

module.exports = GPKManager;
